#!/usr/bin/env node
'use strict';
/**
 * HubOpenAI 远程 Agent
 * JSON-RPC 2.0 over stdio，支持 readFile、readDir、writeFile、exec 等
 * 无第三方依赖，仅 Node 内置模块
 */
const fs = require('fs');
const path = require('path');
const { spawnSync } = require('child_process');
const readline = require('readline');

const rl = readline.createInterface({ input: process.stdin, output: process.stdout, terminal: false });

function send(obj) {
  console.log(JSON.stringify(obj));
}

function safePath(input) {
  if (!input || typeof input !== 'string') return null;
  const resolved = path.resolve(input);
  if (process.platform === 'win32') return resolved;
  if (!resolved.startsWith('/')) return null;
  return resolved;
}

function exists(p) {
  try {
    fs.accessSync(p);
    return true;
  } catch {
    return false;
  }
}

const handlers = {
  ping: () => ({ pong: true }),

  readFile: (p) => {
    const filePath = safePath(p.path);
    if (!filePath) throw new Error('Invalid path');
    const content = fs.readFileSync(filePath, 'utf-8');
    return { content };
  },

  readDir: (p) => {
    const dir = safePath(p.path) || '.';
    const entries = fs.readdirSync(dir, { withFileTypes: true })
      .filter((e) => e.name !== '.' && e.name !== '..')
      .map((e) => ({
        name: e.name,
        longname: e.isDirectory() ? `d--------- ${e.name}` : `- ${e.name}`,
        isDir: e.isDirectory(),
      }));
    return { entries };
  },

  writeFile: (p) => {
    const filePath = safePath(p.path);
    if (!filePath) throw new Error('Invalid path');
    const dir = path.dirname(filePath);
    if (dir && dir !== '.') fs.mkdirSync(dir, { recursive: true });
    fs.writeFileSync(filePath, p.content || '', 'utf-8');
    return { ok: true };
  },

  mkdir: (p) => {
    const dirPath = safePath(p.path);
    if (!dirPath) throw new Error('Invalid path');
    fs.mkdirSync(dirPath, { recursive: p.recursive !== false });
    return { ok: true };
  },

  exists: (p) => {
    const filePath = safePath(p.path);
    if (!filePath) return { exists: false };
    return { exists: exists(filePath) };
  },

  stat: (p) => {
    const filePath = safePath(p.path);
    if (!filePath) throw new Error('Invalid path');
    const s = fs.statSync(filePath);
    return { isDir: s.isDirectory(), size: s.size, mtime: s.mtimeMs };
  },

  delete: (p) => {
    const filePath = safePath(p.path);
    if (!filePath) throw new Error('Invalid path');
    const s = fs.statSync(filePath);
    if (s.isDirectory()) {
      fs.rmSync(filePath, { recursive: true });
    } else {
      fs.unlinkSync(filePath);
    }
    return { ok: true };
  },

  exec: (p) => {
    const cwd = (p.cwd && safePath(p.cwd)) || process.cwd();
    const isWin = process.platform === 'win32';
    const { stdout, stderr, status } = spawnSync(
      isWin ? 'cmd' : 'sh',
      isWin ? ['/c', p.command] : ['-c', p.command],
      { cwd, encoding: 'utf-8' }
    );
    return { stdout: stdout || '', stderr: stderr || '', code: status };
  },
};

rl.on('line', (line) => {
  let req;
  try {
    req = JSON.parse(line);
  } catch {
    return send({ jsonrpc: '2.0', id: null, error: { code: -32700, message: 'Parse error' } });
  }
  const { jsonrpc, id, method, params } = req;
  if (jsonrpc !== '2.0' || !method) {
    return send({ jsonrpc: '2.0', id, error: { code: -32600, message: 'Invalid Request' } });
  }
  const handler = handlers[method];
  if (!handler) {
    return send({ jsonrpc: '2.0', id, error: { code: -32601, message: `Method not found: ${method}` } });
  }
  Promise.resolve(handler(params || {}))
    .then((result) => send({ jsonrpc: '2.0', id, result }))
    .catch((err) => send({ jsonrpc: '2.0', id, error: { code: -32603, message: String(err?.message || err) } }));
});
