import { expect } from "chai";
import { describe, it } from "mocha";

import { OpenAISSEStreamProcessor } from "../../handlers/stream/processors/OpenAISSEStreamProcessor.js";
import { TEST_TIMEOUT_STANDARD } from "../../test/utils/testConfig.js";

import type { Response } from "express";

// align with NodeJS type for gc
declare global {
  var gc: NodeJS.GCFunction | undefined;
}

interface MockResponse {
  write: () => void;
  end: () => void;
  setHeader: () => void;
  headersSent: boolean;
  writableEnded: boolean;
}

describe("Memory Usage Tests", function () {
  this.timeout(TEST_TIMEOUT_STANDARD);

  it("should not leak memory when processing large streams", function () {
    const mockRes: MockResponse = {
      write: () => { },
      end: () => { },
      setHeader: () => { },
      headersSent: false,
      writableEnded: false,
    };

    const processor = new OpenAISSEStreamProcessor(mockRes as unknown as Response);
    processor.setTools([
      { type: 'function', function: { name: "search", parameters: { type: 'object', properties: {} } } },
      { type: 'function', function: { name: "run_code", parameters: { type: 'object', properties: {} } } },
      { type: 'function', function: { name: "analyze", parameters: { type: 'object', properties: {} } } },
    ]);

    const initialMemory = process.memoryUsage().heapUsed;

    for (let i = 0; i < 5000; i++) {
      const contentType = i % 100;
      let content: string;

      if (contentType === 0) {
        content = `<search>`;
      } else if (contentType === 50) {
        content = `</search>`;
      } else if (contentType === 25) {
        content = `<query>Finding memory leaks in Node.js applications</query>`;
      } else if (contentType === 75) {
        content = `I'm thinking about how to solve this problem...`;
      } else if (contentType === 10) {
        content = `${"This is some content that would be generated by an LLM. ".repeat(5)}`;
      } else {
        content = `content chunk ${i}`;
      }

      const chunk = `{"id":"chunk${i}","choices":[{"delta":{"content":"${content}"}}]}`;
      processor.processChunk(Buffer.from(chunk));

      if (i % 1000 === 0 && i > 0) {
        const currentMemory = process.memoryUsage().heapUsed;
        const memoryGrowthMB = (currentMemory - initialMemory) / (1024 * 1024);
        console.log(
          `Memory growth after ${i} chunks: ${memoryGrowthMB.toFixed(2)}MB`,
        );
      }
    }

    processor.end();

    const finalMemory = process.memoryUsage().heapUsed;
    const memoryGrowthMB = (finalMemory - initialMemory) / (1024 * 1024);
    console.log(`Total memory growth: ${memoryGrowthMB.toFixed(2)}MB`);

    expect(memoryGrowthMB).to.be.below(100);

    if (global.gc) {
      global.gc();
      console.log("Garbage collection triggered");
    }
  });

  it("should handle buffer flushing under memory pressure", function () {
    const mockRes: MockResponse = {
      write: () => { },
      end: () => { },
      setHeader: () => { },
      headersSent: false,
      writableEnded: false,
    };

    const processor = new OpenAISSEStreamProcessor(mockRes as unknown as Response);
    processor.setTools([{ type: 'function', function: { name: "search", parameters: { type: 'object', properties: {} } } }]);

    const largeChunk = {
      id: "large-chunk",
      choices: [
        {
          delta: {
            content: `${"A very long text that would normally exceed buffer limits. ".repeat(1000)}<search><query>${"A very long search query. ".repeat(50)}</query></search>`,
          },
        },
      ],
    };

    processor.processChunk(Buffer.from(JSON.stringify(largeChunk)));
    processor.end();

  });
});