import { expect } from "chai";
import { after, before, describe, it } from "mocha";

import { detectPotentialToolCall } from "../../../handlers/toolCallHandler.js";
import { logger } from "../../../logging/index.js";
import { extractToolCall } from "../../../parsers/xml/index.js";

import type { ToolCallDetectionResult, ExtractedToolCall } from "../../../types/index.js";

describe("Debug Tool XML Extraction Tests", function () {
  before(function () {
  (logger as unknown as { level: string }).level = "debug";
  });

  after(function () {
  (logger as unknown as { level: string }).level = "info";
  });

  const htmlToolCall = `<insert_edit_into_file>
  <explanation>Add HTML content to index page</explanation>
  <filePath>/Users/m3hdi/my-project/index.html</filePath>
  <code><!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Website</title>
    <!-- This is a comment with <tags> inside -->
</head>
<body>
    <h1>Welcome to My Website</h1>
    <p>This is an example of HTML content with < and > characters.</p>
    <div class="container">
        <ul>
            <li>Item 1</li>
            <li>Item 2</li>
            <li>Item with x < 10 condition</li>
        </ul>
    </div>
</body>
</html></code>
</insert_edit_into_file>`;

  it("should extract tool call with HTML content correctly", function () {
    const knownToolNames: string[] = ["insert_edit_into_file", "create_file"];

    const detection: ToolCallDetectionResult = detectPotentialToolCall(htmlToolCall, knownToolNames);
    expect(detection).to.not.be.null;
    expect(detection.isPotential).to.be.true;
    expect(detection.mightBeToolCall).to.be.true;
    expect(detection.rootTagName).to.equal("insert_edit_into_file");

    try {
      const safeHtmlToolCall = htmlToolCall
        .replace(
          "<!-- This is a comment with <tags> inside -->",
          "<!-- This is a comment with &lt;tags&gt; inside -->",
        )
        .replace(
          "<li>Item with x < 10 condition</li>",
          "<li>Item with x &lt; 10 condition</li>",
        );

      const parsed: ExtractedToolCall | null = extractToolCall(safeHtmlToolCall, [
        "insert_edit_into_file",
      ]);

      expect(parsed).to.not.be.null;
  expect(parsed).to.not.be.null;
  const p = parsed as ExtractedToolCall;
  expect(p.name).to.equal("insert_edit_into_file");
  expect(p.arguments).to.be.an("object");
  expect((p.arguments as Record<string, unknown>)['explanation']).to.equal(
        "Add HTML content to index page",
      );
      const parsedP = parsed as ExtractedToolCall;
      expect((parsedP.arguments as Record<string, unknown>)['filePath']).to.equal(
        "/Users/m3hdi/my-project/index.html",
      );
      expect((parsedP.arguments as Record<string, unknown>)['code']).to.include("<!DOCTYPE html>");
    } catch (_err: unknown) {
      console.log(
        "XML parsing issues are expected for HTML content - detection still worked",
      );
    }
  });

  it("should handle XML with CDATA sections", function () {
    const xmlWithCDATA = `<insert_edit_into_file>
  <explanation>Add JavaScript code</explanation>
  <filePath>/path/to/file.js</filePath>
  <code><![CDATA[
function compare(a, b) {
  if (a < b) {
    return -1;
  } else if (a > b) {
    return 1;
  }
  return 0;
}
]]></code>
</insert_edit_into_file>`;

    const parsed: ExtractedToolCall | null = extractToolCall(xmlWithCDATA, [
      "insert_edit_into_file",
    ]);

  expect(parsed).to.not.be.null;
  const parsed2 = parsed as ExtractedToolCall;
  expect(parsed2.name).to.equal("insert_edit_into_file");
  expect((parsed2.arguments as Record<string, unknown>)['code']).to.include("if (a < b)");
  });
});