/**
 * Handler Utilities - SSOT for shared request handler functionality
 *
 * Centralizes common logic used across handlers:
 * - Auth header extraction
 * - Debug logging
 * - Success response sending
 */
import { logger } from '../../logging/index.js';
import { configService } from '../../services/index.js';
/**
 * Extract authorization header from request
 * Returns string if present, undefined otherwise
 */
export function extractAuthHeader(req) {
    return typeof req.headers.authorization === 'string'
        ? req.headers.authorization
        : undefined;
}
/**
 * Log response payload if debug mode is enabled
 */
export function logDebugResponse(context, response) {
    if (configService.isDebugMode()) {
        logger.debug(`[${context}] Response payload:`, JSON.stringify(response, null, 2));
    }
}
/**
 * Send JSON success response with optional debug logging
 */
export function sendSuccessJSON(res, data, debugContext) {
    if (debugContext) {
        logDebugResponse(debugContext, data);
    }
    res.status(200).json(data);
}
/**
 * Get backend mode and auth header - common pattern across handlers
 */
export function getBackendContext(req) {
    return {
        backendMode: configService.getBackendMode(),
        authHeader: extractAuthHeader(req),
    };
}
//# sourceMappingURL=handlerUtils.js.map