/**
 * Base Provider Converter Interface
 *
 * Defines the contract that all provider converters must implement
 * for bidirectional conversion to/from the generic schema.
 */
import { hasTools } from '../utils/formatUtils.js';
import { isRecord } from '../utils/typeGuards.js';
// Abstract base class with common functionality
export class BaseConverter {
    // Default implementations
    async validateRequest(request) {
        await Promise.resolve();
        const errors = [];
        if (!isRecord(request)) {
            errors.push("Request is null or undefined");
            return { valid: false, errors };
        }
        if (!Array.isArray(request['messages'])) {
            errors.push("Messages array is required");
        }
        else if (request['messages'].length === 0) {
            errors.push("At least one message is required");
        }
        return { valid: errors.length === 0, errors };
    }
    async checkCompatibility(request) {
        await Promise.resolve();
        const warnings = [];
        const unsupportedFeatures = [];
        const transformations = [];
        if (hasTools(request) && !this.capabilities.toolCalls) {
            unsupportedFeatures.push("toolCalls");
            transformations.push({
                from: "tool_calls",
                to: "text_instructions",
                description: "Tool calls will be converted to text instructions",
            });
        }
        if (request.stream === true && !this.capabilities.streaming) {
            unsupportedFeatures.push("streaming");
            warnings.push("Streaming not supported, will return complete response");
        }
        if (typeof request.n === "number" && request.n > 1 && !this.capabilities.multipleChoices) {
            unsupportedFeatures.push("multipleChoices");
            transformations.push({
                from: "n > 1",
                to: "n = 1",
                description: "Multiple choices not supported, using single response",
            });
        }
        if (request.logprobs === true && !this.capabilities.logprobs) {
            unsupportedFeatures.push("logprobs");
            warnings.push("Log probabilities not available for this provider");
        }
        if (request.responseFormat !== undefined
            && request.responseFormat !== "text"
            && !this.capabilities.jsonMode) {
            unsupportedFeatures.push("jsonMode");
            warnings.push("JSON response format not supported, using text mode");
        }
        if (request.responseFormat !== undefined
            && typeof request.responseFormat === "object"
            && !this.capabilities.structuredOutputs) {
            unsupportedFeatures.push("structuredOutputs");
            transformations.push({
                from: "structured_outputs",
                to: "json_mode",
                description: "Structured outputs converted to JSON mode with instructions",
            });
        }
        return {
            compatible: unsupportedFeatures.length === 0,
            warnings,
            unsupportedFeatures,
            transformations,
        };
    }
    // Utility methods for common operations
    createContext(sourceProvider, targetProvider) {
        return {
            sourceProvider,
            targetProvider: targetProvider ?? this.provider,
            requestId: Math.random().toString(36).slice(2, 11),
            transformationLog: [],
        };
    }
    logTransformation(context, step, description) {
        if (Array.isArray(context.transformationLog)) {
            context.transformationLog.push({
                step,
                description,
                timestamp: Date.now(),
            });
        }
    }
    extractModelFromRequest(request) {
        if (!isRecord(request)) {
            return null;
        }
        const model = request['model'];
        if (typeof model === "string" && model.trim() !== "") {
            return model;
        }
        const deployment = request['deployment'];
        if (typeof deployment === "string" && deployment.trim() !== "") {
            return deployment;
        }
        return null;
    }
    generateId(prefix = "chatcmpl") {
        const timestamp = Math.floor(Date.now() / 1000);
        const random = Math.random().toString(36).slice(2, 14);
        return `${prefix}-${timestamp}-${random}`;
    }
    getCurrentTimestamp() {
        return Math.floor(Date.now() / 1000);
    }
    // Parameter transformation utilities
    transformParameters(params, _direction) {
        return params;
    }
    mapParameters(params, mapping, direction) {
        const result = {};
        if (direction === "toGeneric") {
            for (const [providerKey, value] of Object.entries(params)) {
                const genericKey = Object.keys(mapping).find((key) => mapping[key] === providerKey);
                if (genericKey) {
                    result[genericKey] = value;
                }
                else {
                    result[providerKey] = value;
                }
            }
        }
        else {
            for (const [genericKey, value] of Object.entries(params)) {
                const providerKey = mapping[genericKey];
                if (providerKey) {
                    result[providerKey] = value;
                }
                else {
                    result[genericKey] = value;
                }
            }
        }
        return result;
    }
}
// Registry for storing converter instances
export class ConverterRegistry {
    converters = new Map();
    register(converter) {
        this.converters.set(converter.provider, converter);
    }
    get(provider) {
        return this.converters.get(provider) ?? null;
    }
    getAll() {
        return new Map(this.converters);
    }
    has(provider) {
        return this.converters.has(provider);
    }
    remove(provider) {
        return this.converters.delete(provider);
    }
    clear() {
        this.converters.clear();
    }
    getAvailableProviders() {
        return Array.from(this.converters.keys());
    }
}
// Global converter registry instance
export const converterRegistry = new ConverterRegistry();
// Utility function to get converter safely
export function getConverter(provider) {
    const converter = converterRegistry.get(provider);
    if (!converter) {
        throw new Error(`No converter registered for provider: ${provider}`);
    }
    return converter;
}
//# sourceMappingURL=base.js.map