/**
 * Ollama API Proxy
 * Proxies Ollama-specific endpoints to the backend Ollama server
 *
 * Routing logic:
 * - If backend mode is Ollama: proxies to BACKEND_LLM_BASE_URL (primary backend)
 * - If backend mode is OpenAI: proxies to OLLAMA_BASE_URL (separate Ollama instance for model management)
 */
import { createProxyMiddleware } from "http-proxy-middleware";
import { OLLAMA_EFFECTIVE_BACKEND_URL } from "../config.js";
import { logger } from "../logging/index.js";
import { logRequestDetails } from "../utils/http/proxyLogging.js";
import { buildProxyRequestInfo, collectBackendHeaders, getOriginalUrl, } from "../utils/http/proxyUtils.js";
const ollamaProxyOptions = {
    target: OLLAMA_EFFECTIVE_BACKEND_URL,
    changeOrigin: true,
    // Don't rewrite paths - Ollama uses /api/* directly
    pathRewrite: (path, req) => {
        const original = getOriginalUrl(req, path);
        logger.debug(`\n[OLLAMA PROXY] Proxying: ${original} -> ${OLLAMA_EFFECTIVE_BACKEND_URL}${path}`);
        return path;
    },
    on: {
        proxyReq: (proxyReq, req, _res) => {
            const expressReq = req;
            // Note: req.body is undefined for proxied routes (body not parsed, forwarded raw)
            logRequestDetails("OLLAMA CLIENT REQUEST", {
                method: expressReq.method,
                headers: expressReq.headers,
                body: undefined, // Body forwarded raw by proxy
                ip: expressReq.ip,
                originalUrl: expressReq.originalUrl,
                path: expressReq.path,
            }, expressReq.headers, undefined);
            // Forward authorization header if present
            const clientAuthHeader = expressReq.headers["authorization"];
            if (clientAuthHeader !== undefined) {
                proxyReq.setHeader("authorization", clientAuthHeader);
            }
            // Set content-type for POST requests
            if (expressReq.method === "POST" && expressReq.body !== undefined) {
                proxyReq.setHeader("content-type", "application/json");
            }
            const backendUrl = `${OLLAMA_EFFECTIVE_BACKEND_URL}${proxyReq.path}`;
            const actualBackendHeaders = collectBackendHeaders(proxyReq);
            const proxyRequestInfo = buildProxyRequestInfo(expressReq, backendUrl, undefined);
            logRequestDetails("OLLAMA PROXY REQUEST", proxyRequestInfo, actualBackendHeaders, undefined);
        },
        proxyRes: (proxyRes, req, res) => {
            const expressReq = req;
            const contentType = proxyRes.headers["content-type"];
            logger.debug(`[OLLAMA PROXY RESPONSE] Status: ${proxyRes.statusCode} (${contentType ?? "N/A"}) for ${expressReq.method} ${expressReq.originalUrl}`);
            logger.debug(`[OLLAMA PROXY RESPONSE] Headers received from backend:`);
            logger.debug(JSON.stringify(proxyRes.headers, null, 2));
            // Preserve streaming for endpoints that support it
            if (typeof contentType === "string" &&
                (contentType.includes("application/x-ndjson") || contentType.includes("text/event-stream"))) {
                res.setHeader("Content-Type", contentType);
                res.setHeader("Transfer-Encoding", "chunked");
            }
            // Note: Do not consume the proxyRes stream here as it breaks passthrough
            // The proxy middleware handles forwarding the response to the client
        },
        error: (err, req, res) => {
            const expressReq = req;
            logger.error(`[OLLAMA PROXY] Error proxying ${expressReq.originalUrl}:`, err);
            if (!res.headersSent) {
                if (err.code === "ECONNREFUSED") {
                    res.statusCode = 503;
                    res.setHeader("Content-Type", "application/json");
                    res.end(JSON.stringify({
                        error: "Service Unavailable",
                        message: `Cannot connect to Ollama backend at ${OLLAMA_EFFECTIVE_BACKEND_URL}. Ensure Ollama is running.`,
                    }));
                }
                else {
                    res.statusCode = 502;
                    res.setHeader("Content-Type", "application/json");
                    res.end(JSON.stringify({
                        error: "Bad Gateway",
                        message: `Ollama Proxy Error: ${err.message}`,
                    }));
                }
            }
            else if (!res.writableEnded) {
                res.end();
            }
        },
    },
};
const ollamaProxy = createProxyMiddleware(ollamaProxyOptions);
export default ollamaProxy;
//# sourceMappingURL=ollamaProxy.js.map