import { logger } from "../logging/index.js";
import { FORMAT_OLLAMA, FORMAT_OPENAI } from "./formatDetector.js";
import { FormatConvertingStreamProcessor } from "./stream/formatConvertingStreamProcessor.js";
import { OllamaLineJSONStreamProcessor } from "./stream/processors/OllamaLineJSONStreamProcessor.js";
import { OpenAISSEStreamProcessor } from "./stream/processors/OpenAISSEStreamProcessor.js";
import { WrapperAwareStreamProcessor } from "./stream/wrapperAwareStreamProcessor.js";
const isOllamaChatStreamingRequest = (body) => {
    if (!body || typeof body !== "object") {
        return false;
    }
    const candidate = body;
    return candidate.stream === true && Array.isArray(candidate.messages) && candidate.messages.length > 0;
};
export function setupStreamHandler(backendStream, res, clientRequestFormat = FORMAT_OPENAI, backendFormat = FORMAT_OPENAI, tools = [], streamOptions, clientRequestBody) {
    logger.debug(`[STREAM] Setting up handler: client=${clientRequestFormat}, backend=${backendFormat}`);
    logger.debug(`[STREAM] Tools received:`, tools.length, tools);
    logger.debug(`[STREAM] Stream options:`, streamOptions);
    logger.info(`[STREAM DEBUG] clientFormat=${clientRequestFormat}, backendFormat=${backendFormat}`);
    let processor;
    const shouldUseOllamaSSE = clientRequestFormat === FORMAT_OLLAMA &&
        backendFormat === FORMAT_OPENAI &&
        isOllamaChatStreamingRequest(clientRequestBody);
    if (clientRequestFormat === FORMAT_OPENAI &&
        backendFormat === FORMAT_OPENAI) {
        logger.debug("[STREAM] Using OpenAI SSE processor (OpenAI backend).");
        logger.info("[STREAM DEBUG] Using OpenAI SSE processor");
        const baseProcessor = new OpenAISSEStreamProcessor(res);
        baseProcessor.setStreamOptions(streamOptions);
        processor = new WrapperAwareStreamProcessor(baseProcessor);
        if (processor.setTools) {
            processor.setTools(tools);
        }
    }
    else if (clientRequestFormat === FORMAT_OLLAMA &&
        backendFormat === FORMAT_OLLAMA) {
        logger.debug("[STREAM] Using Ollama Line-JSON processor (Ollama native backend).");
        logger.info("[STREAM DEBUG] Using Ollama Line-JSON processor (Ollama → Ollama)");
        processor = new OllamaLineJSONStreamProcessor(res);
        processor.setStreamOptions?.(streamOptions);
        if (processor.setTools) {
            processor.setTools(tools);
        }
    }
    else {
        logger.debug(`[STREAM] Using format converting processor (${backendFormat} -> ${clientRequestFormat}).`);
        logger.info(`[STREAM DEBUG] Using FormatConvertingStreamProcessor (${backendFormat} → ${clientRequestFormat})`);
        processor = new FormatConvertingStreamProcessor(res, backendFormat, clientRequestFormat, shouldUseOllamaSSE ? { targetStreamMode: 'sse' } : undefined);
        if (processor.setTools) {
            processor.setTools(tools);
        }
    }
    if (processor.pipeFrom) {
        processor.pipeFrom(backendStream);
    }
}
// Add pipeFrom method to stream processors that don't have it
const streamProcessors = [
    OpenAISSEStreamProcessor,
    OllamaLineJSONStreamProcessor,
    FormatConvertingStreamProcessor,
];
streamProcessors.forEach((Processor) => {
    Processor.prototype.pipeFrom ??= function (sourceStream) {
        // Flag to prevent race condition in stream cleanup (double-destroy scenarios)
        let cleanupInProgress = false;
        const safeDestroySourceStream = () => {
            if (cleanupInProgress || sourceStream.destroyed) {
                return;
            }
            cleanupInProgress = true;
            sourceStream.destroy();
        };
        // Handle client disconnect - cleanup backend stream to avoid wasting resources
        if (this.res && !this.res.writableEnded) {
            this.res.on('close', () => {
                logger.debug(`[STREAM] Client disconnected, cleaning up backend stream for ${this.constructor.name}`);
                safeDestroySourceStream();
            });
        }
        sourceStream.on("data", (chunk) => {
            const handleError = (error) => {
                const errorMessage = error instanceof Error ? error.message : 'Unknown error';
                logger.error(`[STREAM] Error processing chunk in ${this.constructor.name}:`, error);
                this.closeStreamWithError(`Error processing stream chunk: ${errorMessage}`);
                safeDestroySourceStream();
            };
            try {
                const result = this.processChunk(chunk);
                if (result && typeof result.then === "function") {
                    result.catch(handleError);
                }
            }
            catch (error) {
                handleError(error);
            }
        });
        sourceStream.on("end", () => {
            try {
                this.end();
            }
            catch (error) {
                const errorMessage = error instanceof Error ? error.message : 'Unknown error';
                logger.error(`[STREAM] Error finalizing stream in ${this.constructor.name}:`, error);
                this.closeStreamWithError(`Error finalizing stream: ${errorMessage}`);
            }
        });
        sourceStream.on("error", (error) => {
            logger.error("[STREAM] Backend stream error:", error);
            this.closeStreamWithError(`Stream error from backend: ${error.message}`);
        });
    };
});
//# sourceMappingURL=streamingHandler.js.map