/**
 * StateTracker - Stream State Management Component
 *
 * SSOT Compliance: Single source of truth for streaming state.
 * Extracted from formatConvertingStreamProcessor to follow KISS principle.
 *
 * Purpose: Track streaming state (tool calls, content emission, flags).
 *
 * Responsibilities:
 * - Track tool call state (in progress, completed)
 * - Track content emission state
 * - Track chunk counts
 * - Provide state queries and updates
 *
 * KISS Compliance: <120 lines, single responsibility, simple state machine
 */
import { logger } from "../../../logging/index.js";
/**
 * StateTracker manages streaming state
 */
export class StateTracker {
    state;
    constructor() {
        this.state = {
            isToolCallInProgress: false,
            hasEmittedContent: false,
            chunkCount: 0,
            streamEnded: false,
            doneSent: false,
        };
    }
    /**
     * Mark tool call as started
     * @param toolName - Name of the tool being called
     */
    startToolCall(toolName) {
        this.state.isToolCallInProgress = true;
        this.state.currentToolCall = toolName;
        logger.debug(`[STATE TRACKER] Tool call started: ${toolName}`);
    }
    /**
     * Mark tool call as ended
     */
    endToolCall() {
        const previousTool = this.state.currentToolCall;
        this.state.isToolCallInProgress = false;
        this.state.currentToolCall = undefined;
        logger.debug(`[STATE TRACKER] Tool call ended: ${previousTool}`);
    }
    /**
     * Record that a chunk was processed
     */
    recordChunk() {
        this.state.chunkCount++;
    }
    /**
     * Record that content was emitted
     */
    recordContent() {
        if (!this.state.hasEmittedContent) {
            this.state.hasEmittedContent = true;
            logger.debug("[STATE TRACKER] First content emitted");
        }
    }
    /**
     * Mark stream as ended
     */
    markStreamEnded() {
        this.state.streamEnded = true;
        logger.debug("[STATE TRACKER] Stream ended");
    }
    /**
     * Mark done signal as sent
     */
    markDoneSent() {
        this.state.doneSent = true;
        logger.debug("[STATE TRACKER] Done signal sent");
    }
    /**
     * Get current state (readonly)
     * @returns Immutable copy of current state
     */
    getState() {
        return { ...this.state };
    }
    /**
     * Check if tool call is in progress
     */
    isToolCallActive() {
        return this.state.isToolCallInProgress;
    }
    /**
     * Check if any content has been emitted
     */
    hasContent() {
        return this.state.hasEmittedContent;
    }
    /**
     * Check if stream has ended
     */
    hasEnded() {
        return this.state.streamEnded;
    }
    /**
     * Check if done signal was sent
     */
    isDoneSent() {
        return this.state.doneSent;
    }
    /**
     * Reset state (for reuse or testing)
     */
    reset() {
        logger.debug("[STATE TRACKER] Resetting state");
        this.state = {
            isToolCallInProgress: false,
            hasEmittedContent: false,
            chunkCount: 0,
            streamEnded: false,
            doneSent: false,
        };
    }
}
//# sourceMappingURL=StateTracker.js.map